<?php
/**
 * Clase para comunicación con la API del Panel SEO
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Panel_API {
    
    private $api_url;
    private $api_key;
    
    public function __construct() {
        $this->api_url = SEO_PANEL_API_URL;
        $this->api_key = get_option('seo_panel_api_key', '');
    }
    
    /**
     * Verificar suscripción activa
     */
    public function check_subscription() {
        $subscription = new SEO_Panel_Subscription();
        return $subscription->is_active();
    }
    
    /**
     * Analizar una URL o archivo
     */
    public function analyze($website, $info = '') {
        if (!$this->check_subscription()) {
            return array(
                'success' => false,
                'error' => 'Suscripción inactiva. Por favor, activa tu suscripción.'
            );
        }
        
        $response = wp_remote_post($this->api_url . '/api/analyze', array(
            'timeout' => 90,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $this->api_key
            ),
            'body' => json_encode(array(
                'website' => $website,
                'info' => $info
            )),
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('SEO Panel API Error: ' . $error_message);
            // Mejorar mensajes de error comunes
            if (strpos($error_message, 'timeout') !== false || strpos($error_message, 'timed out') !== false) {
                $error_message = 'La solicitud tardó demasiado tiempo. La URL puede estar muy lenta o inaccesible. Intenta con una URL diferente.';
            } elseif (strpos($error_message, 'Connection') !== false || strpos($error_message, 'Could not resolve') !== false) {
                $error_message = 'No se pudo conectar con el servidor API (' . $this->api_url . '). Verifica que la URL sea correcta y que el servidor esté disponible.';
            } elseif (strpos($error_message, 'SSL') !== false || strpos($error_message, 'certificate') !== false) {
                $error_message = 'Error de certificado SSL. Verifica la configuración del servidor.';
            }
            return array(
                'success' => false,
                'error' => $error_message
            );
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            $body = wp_remote_retrieve_body($response);
            error_log('SEO Panel API: Código de respuesta ' . $code . ' - Body: ' . substr($body, 0, 200));
            return array(
                'success' => false,
                'error' => 'Error del servidor: Código ' . $code . '. Verifica tu API Key y que la suscripción esté activa.'
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('SEO Panel API: Error parseando JSON - ' . json_last_error_msg());
            return array(
                'success' => false,
                'error' => 'Error al procesar la respuesta del servidor: ' . json_last_error_msg()
            );
        }
        
        if (isset($data['error'])) {
            error_log('SEO Panel API: Error en respuesta: ' . $data['error']);
            return array(
                'success' => false,
                'error' => $data['error']
            );
        }
        
        return $data;
    }
    
    /**
     * Analizar un archivo específico
     */
    public function analyze_file($file_path, $info = '') {
        if (!$this->check_subscription()) {
            return array(
                'success' => false,
                'error' => 'Suscripción inactiva.'
            );
        }
        
        $response = wp_remote_post($this->api_url . '/api/analyze_file', array(
            'timeout' => 90,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $this->api_key
            ),
            'body' => json_encode(array(
                'file_path' => $file_path,
                'info' => $info
            )),
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }
    
    /**
     * Aplicar mejoras SEO
     */
    public function apply_seo($file_path, $seo_data) {
        if (!$this->check_subscription()) {
            return array(
                'success' => false,
                'error' => 'Suscripción inactiva.'
            );
        }
        
        $response = wp_remote_post($this->api_url . '/api/apply_seo', array(
            'timeout' => 90,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $this->api_key
            ),
            'body' => json_encode(array(
                'file_path' => $file_path,
                'seo_data' => $seo_data
            )),
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }
    
    /**
     * Verificar API key y conexión
     */
    public function verify_api_key($api_key = '') {
        if (empty($api_key)) {
            $api_key = $this->api_key;
        }
        
        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => 'API Key no configurada'
            );
        }
        
        $response = wp_remote_get($this->api_url . '/api/verify', array(
            'timeout' => 30,
            'headers' => array(
                'X-API-Key' => $api_key
            )
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'Error de conexión: ' . $response->get_error_message()
            );
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($code === 200 && isset($data['valid']) && $data['valid']) {
            return array(
                'success' => true,
                'valid' => true,
                'connected' => isset($data['connected']) ? $data['connected'] : true,
                'user' => isset($data['user']) ? $data['user'] : '',
                'plan' => isset($data['plan']) ? $data['plan'] : '',
                'message' => isset($data['message']) ? $data['message'] : 'Conexión exitosa'
            );
        }
        
        return array(
            'success' => false,
            'valid' => false,
            'message' => isset($data['message']) ? $data['message'] : 'API Key inválida'
        );
    }
    
    /**
     * Verificar conexión (alias para compatibilidad)
     */
    public function check_connection() {
        return $this->verify_api_key();
    }
}
