<?php
/**
 * Clase para funcionalidades públicas
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Panel_Public {
    
    public function __construct() {
        add_action('wp_ajax_seo_panel_analyze', array($this, 'ajax_analyze'));
        add_action('wp_ajax_seo_panel_apply', array($this, 'ajax_apply'));
        add_action('wp_ajax_seo_panel_verify_key', array($this, 'ajax_verify_key'));
    }
    
    public function ajax_verify_key() {
        check_ajax_referer('seo_panel_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permisos insuficientes'));
            return;
        }
        
        $api = new SEO_Panel_API();
        $api_key = get_option('seo_panel_api_key', '');
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => 'No hay API key configurada'));
            return;
        }
        
        $result = $api->verify_api_key($api_key);
        
        if (isset($result['success']) && $result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    public function ajax_analyze() {
        // Log para debugging
        error_log('SEO Panel: ajax_analyze llamado');
        
        check_ajax_referer('seo_panel_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            error_log('SEO Panel: Permisos insuficientes');
            wp_send_json_error(array('message' => 'Permisos insuficientes'));
            return;
        }
        
        $url = sanitize_text_field($_POST['url']);
        $info = sanitize_textarea_field($_POST['info']);
        
        error_log('SEO Panel: Analizando URL: ' . $url);
        
        $api = new SEO_Panel_API();
        $result = $api->analyze($url, $info);
        
        error_log('SEO Panel: Resultado de API: ' . print_r($result, true));
        
        if (isset($result['error'])) {
            error_log('SEO Panel: Error en análisis: ' . $result['error']);
            wp_send_json_error(array('message' => $result['error']));
            return;
        }
        
        if (!isset($result['analysis']) && !isset($result['seo_content'])) {
            error_log('SEO Panel: Resultado inválido - no tiene analysis ni seo_content');
            wp_send_json_error(array('message' => 'Respuesta inválida del servidor. Verifica la configuración de la API.'));
            return;
        }
        
        // Guardar análisis
        $this->save_analysis($url, $result);
        
        // Generar HTML de resultados
        $html = $this->render_results($result);
        
        error_log('SEO Panel: Análisis completado exitosamente');
        wp_send_json_success(array('html' => $html, 'data' => $result));
    }
    
    public function ajax_apply() {
        check_ajax_referer('seo_panel_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permisos insuficientes'));
            return;
        }
        
        $file_path = sanitize_text_field($_POST['file_path']);
        $seo_data = $_POST['seo_data'];
        
        $api = new SEO_Panel_API();
        $result = $api->apply_seo($file_path, $seo_data);
        
        wp_send_json($result);
    }
    
    private function save_analysis($url, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'seo_panel_analyses';
        
        // Intentar obtener post_id de la URL
        $post_id = url_to_postid($url);
        if (!$post_id && is_numeric($url)) {
            $post_id = intval($url);
        }
        
        $wpdb->insert(
            $table,
            array(
                'post_id' => $post_id ?: 0,
                'analysis_data' => json_encode($data),
                'seo_score' => $this->calculate_score($data)
            ),
            array('%d', '%s', '%d')
        );
    }
    
    private function calculate_score($data) {
        $score = 100;
        if (isset($data['analysis']['issues'])) {
            $score -= count($data['analysis']['issues']) * 10;
        }
        return max(0, min(100, $score));
    }
    
    private function render_results($data) {
        ob_start();
        ?>
        <div class="seo-analysis-results">
            <?php if (isset($data['analysis'])): ?>
                <h3>Análisis Actual</h3>
                <p><strong>Título:</strong> <?php echo esc_html($data['analysis']['title']); ?></p>
                <p><strong>Meta Descripción:</strong> <?php echo esc_html($data['analysis']['meta_description']); ?></p>
                
                <?php if (!empty($data['analysis']['issues'])): ?>
                    <h4>Problemas:</h4>
                    <ul>
                        <?php foreach ($data['analysis']['issues'] as $issue): ?>
                            <li><?php echo esc_html($issue); ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            <?php endif; ?>
            
            <?php if (isset($data['seo_content'])): ?>
                <h3>Mejoras Sugeridas</h3>
                <p><strong>Título optimizado:</strong> <?php echo esc_html($data['seo_content']['title']); ?></p>
                <p><strong>Meta descripción optimizada:</strong> <?php echo esc_html($data['seo_content']['meta_description']); ?></p>
                <p><strong>H1 optimizado:</strong> <?php echo esc_html($data['seo_content']['h1']); ?></p>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
}
